﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Input;

namespace WpfApp1.Common
{
    // 非泛型版本（兼容原有用法）
    public class RelayCommand : ICommand
    {
        private readonly Action<object?> _execute;
        private readonly Func<object?, bool>? _canExecute;

        public RelayCommand(Action<object?> execute, Func<object?, bool>? canExecute = null)
        {
            _execute = execute ?? throw new ArgumentNullException(nameof(execute));
            _canExecute = canExecute;
        }

        public bool CanExecute(object? parameter) => _canExecute?.Invoke(parameter) ?? true;

        public void Execute(object? parameter) => _execute(parameter);

        public event EventHandler? CanExecuteChanged
        {
            add => CommandManager.RequerySuggested += value;
            remove => CommandManager.RequerySuggested -= value;
        }
    }

    // 泛型版本（支持指定参数类型，避免手动强转）
    public class RelayCommand<T> : ICommand
    {
        private readonly Action<T?> _execute;
        private readonly Func<T?, bool>? _canExecute;

        public RelayCommand(Action<T?> execute, Func<T?, bool>? canExecute = null)
        {
            _execute = execute ?? throw new ArgumentNullException(nameof(execute));
            _canExecute = canExecute;
        }

        // 适配ICommand的非泛型CanExecute（自动转换参数类型）
        public bool CanExecute(object? parameter)
        {
            // 处理参数类型不匹配的情况（比如传了null或错误类型）
            if (parameter is not T && parameter is not null)
                return false;
            return _canExecute?.Invoke((T?)parameter) ?? true;
        }

        // 适配ICommand的非泛型Execute（自动转换参数类型）
        public void Execute(object? parameter)
        {
            if (parameter is T typedParameter)
                _execute(typedParameter);
            else if (parameter is null && default(T) is null)
                _execute(default); // 处理null参数（比如T是引用类型）
        }

        public event EventHandler? CanExecuteChanged
        {
            add => CommandManager.RequerySuggested += value;
            remove => CommandManager.RequerySuggested -= value;
        }
    }
}
