﻿using CommunityToolkit.Mvvm.ComponentModel;
using CommunityToolkit.Mvvm.Input;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Text;
using System.Windows.Input;
using WpfApp1.Model;

namespace WpfApp1.ViewModel
{

    // 改写后：继承 ObservableObject（框架内置，替代手搓的 ViewModelBase）
    public partial class TodoViewModel : ObservableObject
    {
        // 改写后：使用 [ObservableProperty] 特性，编译时自动生成以下内容：
        // 1. 私有字段 _newTodoText
        // 2. 公共属性 NewTodoText（包含 get/set，且 set 时自动触发 PropertyChanged 通知）
        // 3. 无需手动调用 OnPropertyChanged()
        [ObservableProperty]
        private string _newTodoText = string.Empty;

        // 当 NewTodoText 变化时，通知 AddTodoCommand 刷新 CanExecute
        partial void OnNewTodoTextChanged(string value)
        {
            AddTodoCommand.NotifyCanExecuteChanged();
        }

        // 待办列表（核心：ObservableCollection，保持不变，框架不替代集合本身）
        public ObservableCollection<TodoItem> TodoItems { get; } = new();

        // 未完成待办数量（计算属性，保持逻辑不变，仅触发通知的方式简化）
        public int UnfinishedCount => TodoItems.Count(t => !t.IsCompleted);

        // 改写后：使用 [RelayCommand] 特性，编译时自动生成以下内容：
        // 1. 公共 ICommand 属性 AddTodoCommand
        // 2. 无需手动初始化 RelayCommand，框架自动绑定对应的执行方法
        // 3. 支持配套的 CanAddTodo 方法（命名规范：命令方法名 + CanExecute），自动作为 CanExecute 逻辑
        [RelayCommand(CanExecute = nameof(CanAddTodo))]
        private void AddTodo()
        {
            var newTodo = new TodoItem { Content = NewTodoText.Trim(), IsCompleted = false };
            TodoItems.Add(newTodo);
            NewTodoText = string.Empty; // 清空输入框（赋值时，框架自动触发 PropertyChanged）
        }

        // AddTodo 命令的 CanExecute 逻辑（命名规范：命令方法名 + CanExecute）
        private bool CanAddTodo()
        {
            return !string.IsNullOrWhiteSpace(NewTodoText);
        }

        // 改写后：泛型命令，[RelayCommand] 自动支持参数类型，无需手动写 RelayCommand<T>
        [RelayCommand]
        private void DeleteTodo(TodoItem? todo)
        {
            if (todo != null && TodoItems.Contains(todo))
            {
                TodoItems.Remove(todo);
            }
        }

        // 改写后：无参数命令，框架自动生成 ClearCompletedCommand
        [RelayCommand]
        private void ClearCompleted()
        {
            var completedItems = TodoItems.Where(t => t.IsCompleted).ToList();
            foreach (var item in completedItems)
            {
                TodoItems.Remove(item);
            }
        }

        public TodoViewModel()
        {
            // 监听 TodoItems 集合变更，更新 UnfinishedCount
            // 仅需手动触发计算属性的通知，框架无额外简化（集合变更本身是 ObservableCollection 的职责）
            TodoItems.CollectionChanged += (s, e) => OnPropertyChanged(nameof(UnfinishedCount));
        }
    }


    /* 下面是手搓代码
    public class TodoViewModel : ViewModelBase // 复用之前的ViewModelBase
    {
        // 绑定输入框的待办内容
        private string _newTodoText;
        public string NewTodoText
        {
            get => _newTodoText;
            set { _newTodoText = value; OnPropertyChanged(); }
        }

        // 待办列表（核心：ObservableCollection）
        public ObservableCollection<TodoItem> TodoItems { get; } = new();

        // 未完成待办数量（计算属性，实时更新）
        public int UnfinishedCount
        {
            get => TodoItems.Count(t => !t.IsCompleted);
        }

        // 命令：添加待办
        public ICommand AddTodoCommand { get; }
        // 命令：删除待办
        public ICommand DeleteTodoCommand { get; }
        // 命令：清空已完成
        public ICommand ClearCompletedCommand { get; }

        public TodoViewModel()
        {
            // 初始化命令
            AddTodoCommand = new RelayCommand(AddTodo, CanAddTodo);
            DeleteTodoCommand = new RelayCommand<TodoItem>(DeleteTodo);
            ClearCompletedCommand = new RelayCommand(ClearCompleted);

            // 监听TodoItems集合变更，更新UnfinishedCount
            TodoItems.CollectionChanged += (s, e) => OnPropertyChanged(nameof(UnfinishedCount));
            // 监听每个TodoItem的IsCompleted变更，更新UnfinishedCount
            // （需额外处理：新增TodoItem时订阅其PropertyChanged事件）
        }

        // AddTodoCommand的执行逻辑
        private void AddTodo(object? parameter)
        {
            var newTodo = new TodoItem { Content = NewTodoText.Trim(), IsCompleted = false };
            TodoItems.Add(newTodo);
            NewTodoText = string.Empty; // 清空输入框
        }

        // AddTodoCommand的CanExecute逻辑（输入为空时禁用）
        private bool CanAddTodo(object? parameter)
        {
            return !string.IsNullOrWhiteSpace(NewTodoText);
        }

        // DeleteTodoCommand的执行逻辑
        private void DeleteTodo(TodoItem? todo)
        {
            if (todo != null && TodoItems.Contains(todo))
            {
                TodoItems.Remove(todo);
            }
        }

        // ClearCompletedCommand的执行逻辑
        private void ClearCompleted(object? parameter)
        {
            var completedItems = TodoItems.Where(t => t.IsCompleted).ToList();
            foreach (var item in completedItems)
            {
                TodoItems.Remove(item);
            }
        }
    }
    */

}
